/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.benchmark;

import java.nio.file.Path;
import java.util.concurrent.TimeUnit;
import org.apache.ignite.internal.sql.engine.util.TpcTable;
import org.apache.ignite.internal.sql.engine.util.tpch.TpchTables;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.infra.Blackhole;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

/**
 * Benchmark that runs sql queries on TPC-H dataset via embedded client.
 */
@State(Scope.Benchmark)
@Fork(1)
@Threads(1)
@Warmup(iterations = 10, time = 2)
@Measurement(iterations = 20, time = 2)
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
@SuppressWarnings({"WeakerAccess", "unused"})
public class SqlSelectAllBenchmark extends AbstractTpcBenchmark {
    /*
        Minimal configuration of this benchmark requires specifying pathToDataset. Latest known location
        of dataset is https://github.com/cmu-db/benchbase/tree/main/data/tpch-sf0.01 for scale factor 0.01
        and https://github.com/cmu-db/benchbase/tree/main/data/tpch-sf0.1 for scale factor 0.1. Dataset
        is set of CSV files with name `{$tableName}.tbl` per each table and character `|` as separator.

        By default, cluster's work directory will be created as a temporary folder. This implies,
        that all data generated by benchmark will be cleared automatically. However, this also implies
        that cluster will be recreated on EVERY RUN. Given there are 25 queries, it results in 25 schema
        initialization and data upload cycles. To initialize cluster once and then reuse it state override
        `AbstractMultiNodeBenchmark.workDir()` method. Don't forget to clear that directory afterwards.
     */

    @Override
    TpcTable[] tablesToInit() {
        return new TpcTable[] { TpchTables.LINEITEM };
    }

    @Override
    Path pathToDataset() {
        throw new RuntimeException("Provide path to directory containing <table_name>.tbl files");
    }

    @Param({
            "IDENTITY", "TRIMMING", "RESHUFFLING"
    })
    private Projection projection;

    /** Benchmark that measures performance of queries from TPC-H suite. */
    @Benchmark
    public void run(Blackhole bh) {
        try (var rs = sql.execute(null, projection.query())) {
            while (rs.hasNext()) {
                bh.consume(rs.next());
            }
        }
    }

    /**
     * Benchmark's entry point.
     */
    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(".*\\." + SqlSelectAllBenchmark.class.getSimpleName() + ".*")
                .build();

        new Runner(opt).run();
    }

    /**
     * Projection types for the benchmark.
     *
     * <p>IDENTITY - selects all columns in the original order.
     * TRIMMING - selects all columns except few ones (preserving related order).
     * RESHUFFLING - selects all columns in a different order.
     */
    public enum Projection {
        IDENTITY {
            @Override
            public String query() {
                StringBuilder sb = new StringBuilder("SELECT ")
                        .append(TpchTables.LINEITEM.columnName(0));

                for (int i = 1; i < TpchTables.LINEITEM.columnsCount(); i++) {
                    sb.append(", ").append(TpchTables.LINEITEM.columnName(i));
                }
                return sb.append(" FROM lineitem").toString();
            }
        },
        TRIMMING {
            @Override
            public String query() {
                StringBuilder sb = new StringBuilder("SELECT ")
                        .append(TpchTables.LINEITEM.columnName(0));

                for (int i = 2; i < TpchTables.LINEITEM.columnsCount(); i++) {
                    sb.append(", ").append(TpchTables.LINEITEM.columnName(i));
                }
                return sb.append(" FROM lineitem").toString();
            }
        },
        RESHUFFLING {
            @Override
            public String query() {
                StringBuilder sb = new StringBuilder("SELECT ")
                        .append(TpchTables.LINEITEM.columnName(1))
                        .append(", ").append(TpchTables.LINEITEM.columnName(0));

                for (int i = 2; i < TpchTables.LINEITEM.columnsCount(); i++) {
                    sb.append(", ").append(TpchTables.LINEITEM.columnName(i));
                }
                return sb.append(" FROM lineitem").toString();
            }
        };

        abstract String query();
    }
}
